
/* Public Domain */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <string.h>

struct tabletype {
	const char * sign;
	const char ** keywords; /* array of keywords, NULL-terminated */
	const char * description; /* ASCIZ description */
	const char ** names;
		/* names array (NULL = special, empty string = unknown */
	unsigned flagsamount; /* amount of names */
	const char ** extnames; /* extnames array, NULL-terminated */
	unsigned offset; /* first lCFG byte offset */
	unsigned bytes; /* how many lCFG bytes (1 or 2) */
	unsigned types; /* Non-Zero if types, Zero if flags */
	const char *** mode; /* mode array of array of pointers */
};

const char* names_progress[] = { "None", "Dots", "Percentage",
	"Bar", "Bar with percentage" };

/* first string is description,
   second is preferred keyword,
   then alternative keywords may follow.
   NULL-terminated */
const char* modekeywords_progress_app[] = {
	"Application", "APP", "APPLICATION", NULL
};
const char* modekeywords_progress_dev[] = {
	"Device driver", "DEV", "DEVICE", NULL
};
const char* modekeywords_progress_boot[] = {
	"Bootloaded", "BOOT", NULL
};
/* points to array of above. NULL-terminated */
const char** modekeywords_progress[] = {
	modekeywords_progress_app,
	modekeywords_progress_dev,
	modekeywords_progress_boot,
	NULL
};
const char * keywords_progress[] = {
	"PROGRESS", "PRO", NULL
};

const char* names_debug[] = { "Check", "Assume",
	"Check only valid", "Check only IISP" };
const char * keywords_debug[] = {
	"DEBUG", "DEB", NULL
};

const char* names_verbose[] = {
	"Intro line",
	"XMS use",
	"CDS writes",
};

const char * keywords_verbose[] = {
	"VERBOSE", "VER", NULL
};

  const char* names_scan[] = {
  	"Ext5 setting", NULL,
  	"Ext15 setting", NULL,
  	"Ext85h setting", NULL,
  	"Ignore subsequent nested",
  	"Ignore subsequent logical",
  	"Ignore subsequent primary",
  	"Traditional order logical",
  	"Active priority",
  	"Subsequent primary after",
  	"",
  	"Ignore FAT32",
  	"Ignore LBA",
  	"DLA sort disable",
	};
/*
mode_ignore_subsequent_nested	equ            1_00_00_00b
mode_ignore_subsequent_logical	equ           10_00_00_00b
mode_ignore_subsequent_primary	equ          100_00_00_00b
mode_traditional_order_logical	equ         1000_00_00_00b
mode_active_priority		equ       1_0000_00_00_00b
mode_subsequent_primary_after	equ      10_0000_00_00_00b
mode_disable_fat32		equ 00_1000_0000_00_00_00b
mode_disable_lba		equ 01_0000_0000_00_00_00b
mode_dlasort_zero		equ 10_0000_0000_00_00_00b
*/
  const char* extnames_scan[] = {
        "0 = follow all",
        "1 = ignore all",
        "2 = follow first of any ext",
        "3 = follow first of this type",
        NULL,
        };

const char * keywords_scan[] = {
	"SCANMODE", "SCAN", "MODE", "SCA", NULL
};

const char* names_haltflag[] = {
	"Don't halt on TAGCRASH",
	"Don't halt on MCB corrupted",
	"Don't halt on HMCB alloc corrupted",
	"Don't halt on HMCB free corrupted",
};

const char* names_menudefault[] = {
	"No menu default",
	"DOS",
	"Debugger",
	"Loader",
};

const char * keywords_haltflag[] = {
	"HALTFLAG", "HALTFLAGS", "HALT", NULL
};

const char * keywords_timeout[] = {
	"TIMEOUT", "TIME", NULL
};

const char * keywords_menudefault[] = {
	"MENUDEFAULT", "MENU", "DEFAULT", NULL
};

const char * keywords_list[] = {
	"LIST", NULL
};

struct tabletype table[] = {
	{ "", keywords_list, "List all", },
	{ "\xEB\xFFlCFG00", keywords_progress, "lDOS inicomp progress choice",
	names_progress, (sizeof(names_progress) / sizeof(names_progress[0])),
	NULL, 16, 1, 1 /* types */, modekeywords_progress },
	{ "\xEB\xFFlCFG00", keywords_debug, "lMS-DOS / lEDR-DOS debugger check option",
	names_debug, (sizeof(names_debug) / sizeof(names_debug[0])),
	NULL, 20, 1 },
	{ "\xEB\xFFlCFG00", keywords_verbose, "lMS-DOS verbosity option",
	names_verbose, (sizeof(names_verbose) / sizeof(names_verbose[0])),
	NULL, 21, 1 },
	{ "\xEB\xFFlCFG00", keywords_scan, "lMS-DOS partition table scan mode",
	names_scan, (sizeof(names_scan) / sizeof(names_scan[0])),
	extnames_scan, 22, 2 },
	{ "\xEB\xFFlCFG00", keywords_haltflag, "lMS-DOS halt flag option",
	names_haltflag, (sizeof(names_haltflag) / sizeof(names_haltflag[0])),
	NULL, 26, 2 },
	{ "MCPCFG""00", keywords_timeout, "lDOS MCP timeout",
	NULL, 0,
	NULL, 16, 1, 2 /* types */ },
	{ "MCPCFG""00", keywords_menudefault, "lDOS MCP menu default",
	names_menudefault, (sizeof(names_menudefault) / sizeof(names_menudefault[0])),
	NULL, 17, 1, 3 /* types */ },
	{ NULL }
};

char cap(char cc) {
  if (cc >= 'a' && cc <= 'z') {
    return cc - 'a' + 'A';
  }
  return cc;
}

unsigned comparecaps(const char * ss, const char * ddallcaps) {
  unsigned ii = 0;
  while (ss[ii] && cap(ss[ii]) == ddallcaps[ii]) {
    ++ ii;
  }
  return (0 != ss[ii] || 0 != ddallcaps[ii]);
}

unsigned parsenumber(char * pp,
	unsigned int replaceforbidden,
	unsigned long int * pwant,
	unsigned long int * pwantreplace,
	unsigned long int * pwantplus,
	unsigned long int * pwantminus) {
    char *originalpp = pp;
    char *expectedend;
    char *parsedend;
    char replace_plus_minus = 0;
    unsigned int base = 10, ii;
    unsigned long int vv = 0;
    if (pp[0] == '+') {
      ++ pp;
      replace_plus_minus = 1;
    } else if (pp[0] == '-') {
      ++ pp;
      replace_plus_minus = 2;
    } else {
      *pwantreplace = 1;
    }
    ii = strlen(pp);
    expectedend = &pp[ii];
    if (ii && (pp[ii - 1] == 'H' || pp[ii - 1] == 'h') ) {
      base = 16;
      -- expectedend;
    } else if (ii >= 2 && pp[0] == '0' && (pp[1] == 'B' || pp[1] == 'b') ) {
      pp += 2;
      base = 2;
    } else if (ii >= 2 && pp[0] == '0' && (pp[1] == 'X' || pp[1] == 'x') ) {
      pp += 2;
      base = 16;
    } else if (ii && (pp[ii - 1] == 'B' || pp[ii - 1] == 'b') ) {
      base = 2;
      -- expectedend;
    }
    vv = strtoul(pp, &parsedend, base);
    if ((0 == vv && pp[0] != '0')
      || vv < 0 || vv > 255
      || expectedend != parsedend) {
      fprintf(stderr,"Error: Invalid choice: %s\n", originalpp);
      return 4;
    }
    switch (replace_plus_minus) {
    case 0:
      if (replaceforbidden) {
        fprintf(stderr,"Error: Invalid choice: %s\n", originalpp);
        return 4;
      }
      *pwant = vv;
      *pwantplus = 0;
      *pwantminus = 0;
      break;
    case 1:
      *pwantplus |= vv;
      *pwantminus &= ~vv;
      break;
    case 2:
      *pwantminus |= vv;
      *pwantplus &= ~vv;
      break;
    }
  return 0;
}

unsigned comparewild(const uint8_t * data, const char * pattern, unsigned length) {
  unsigned ii = 0;
  for (; (data[ii] == (uint8_t)pattern[ii]
      || (uint8_t)pattern[ii] == 255)
    && ii < length; ++ii);
  return ii != length;
}

unsigned isnumber(const char cc) {
  return cc >= '0' && cc <= '9';
}

uint32_t getu32(uint8_t const * pp) {
  return (uint32_t)pp[0]
	+ ((uint32_t)pp[1] << 8)
	+ ((uint32_t)pp[2] << 16)
	+ ((uint32_t)pp[3] << 24);
}

uint16_t getu16(uint8_t const * pp) {
  return (uint16_t)pp[0]
	+ ((uint16_t)pp[1] << 8);
}

#define FROMPATLDOS
#include "patchqry.c"

int main(int argc, char **argv) {
  FILE* ff;
  const unsigned int buffersize = 8192;
  const char ** names;
  const char ** extnames;
  const char * choicesstring;

  unsigned int ii, jj, offset = 0, offsetmod8, offsetdiv8,
  	bytes = 0, bytesmask, size, mode, firstcurrent,
	choices, query = 1, found = 0, at, current, flagsamount, index = 0,
	list = 0, first = 1, listpro = 0, listfoundany = 0,
	whichblock = 0, currentearly = 1, listonlypro = 0,
	hintshown = 0, allblocks = 0, moreblocks = 0,
	initialindex, firstblocks = 1, dosepbeforehint = 0,
	dosepbeforelist = 0, listfoundanyone = 0, listfoundanycfg = 0;
  unsigned long int want = 0, wantreplace = 0, wantminus = 0, wantplus = 0;
  uint8_t buffer[buffersize];
  uint32_t bufferbase = 0;
  uint16_t amountblock = 1, bufferlength;
  if (argc >= 2 && isnumber(argv[currentearly][0])) {
    for (found = 0, ii = 0; argv[currentearly][ii]; ++ ii) {
      if (! isnumber(argv[currentearly][ii])) {
        found = 1;
      }
    }
    if (! found) {
      whichblock = strtoul(argv[currentearly], NULL, 10);
      ++ currentearly;
    }
  }
  if (currentearly == 1 && argc >= 2 && ! comparecaps(argv[currentearly], "ALL")) {
    allblocks = 1;
    ++ currentearly;
  }
  if (currentearly == 1 && argc >= 2
  && (! comparecaps(argv[currentearly], "QUERY")
    || ! comparecaps(argv[currentearly], "QRY"))) {
    return patchqrymain(argc - 1, argv + 1);
  }
  if (argc <= currentearly) {
    printf("Usage: patldos [number] KEYWORD\n\n");
    printf("Supported keywords:\n");
    for (ii = 0; table[ii].keywords; ++ ii) {
      const char * sep = "";
      printf("%s: ", table[ii].description);
      for (jj = 0; table[ii].keywords[jj]; ++ jj) {
        printf("%s%s", sep, table[ii].keywords[jj]);
        sep = ", ";
      }
      printf("\n");
    }
    printf("lDOS query patch: QUERY, QRY\n");
    return 0;
  }
  for (found = 0, index = 0; table[index].keywords; ++ index) {
    for (jj = 0; table[index].keywords[jj]; ++ jj) {
      if (! comparecaps(argv[currentearly], table[index].keywords[jj])) {
        found = 1;
        break;
      }
    }
    if (found) {
      break;
    }
  }
  if (! found) {
    fprintf(stderr,"Error: Invalid keyword: %s\n", argv[currentearly]);
    return 1;
  }
  if (! index) {
    list = 1;
    ++ index;
  }
initialindex = index;
while (firstblocks || (allblocks && moreblocks)) {
firstblocks = 0;
index = initialindex;
listpro = 0;
bufferbase = 0;
bufferlength = buffersize;
hintshown = 0;
first = 1;
for (; (first || list || listonlypro) && table[index].keywords; ++ index) {
  first = 0;
  if ((list || listonlypro) && index == 2) {
    ++ listpro;
    if (listpro < 3) {
      -- index;
    } else if (listonlypro) {
      break;
    }
  }
  flagsamount = table[index].flagsamount;
  names = table[index].names;
  extnames = table[index].extnames;
  if (! table[index].types) {
    choicesstring = "flag";
  } else if (table[index].types == 1) {
    choicesstring = "progress choice";
  } else if (table[index].types == 2) {
    choicesstring = "timeout";
  } else if (table[index].types == 3) {
    choicesstring = "menu default choice";
  } else {
    choicesstring = "flag";
  }
  if (argc < (2 + currentearly)) {
    if (list && index > 1) {
      printf("\n");
    }
    if (! table[index].mode) {
      printf("Usage: patldos [%s [ldos.com [number]]]\n\n",
    	table[index].keywords[0]);
    } else {
      const char * sep = "";
      printf("Usage: patldos [%s [ldos.com [",
    	table[index].keywords[0]);
      for (jj = 0; table[index].mode[jj]; ++ jj) {
        printf("%s%s", sep, table[index].mode[jj][1]);
        sep = "|";
      }
      printf(" [number]]]]\n\n");
    }
    printf("Supported %ss:\n", choicesstring);
    if (table[index].types == 2) {
      printf(	"     0: Immediate timeout\n"
		" 1-254: Timeout after given number of seconds\n"
		"   255: Infinite timeout\n");
    }
    for (ii = 0; ii < flagsamount; ++ ii) {
      if (ii + 1 < flagsamount && !names[ii + 1]) {
        printf(" %5u|%u: %s\n", 1 << ii, 1 << (ii + 1), names[ii]);
      } else if (!names[ii]) {
      } else if (!*names[ii]) {
      } else if (table[index].types) {
        printf(" %5u: %s\n", ii, names[ii]);
      } else if ((1 << ii) > 8) {
        printf(" %4Xh=%5u: %s\n", 1 << ii, 1 << ii, names[ii]);
      } else {
        printf(" %5u: %s\n", 1 << ii, names[ii]);
      }
    }
    if (extnames) {
      printf("Supported ext settings:\n");
      for (ii = 0; extnames[ii]; ++ ii) {
        printf(" %s\n", extnames[ii]);
      }
    }
    if (list) {
      listpro = 3;
      continue;
    }
    return 0;
  }
  ff = fopen(argv[currentearly + 1], "r+b");
  if (! ff) {
    fprintf(stderr,"Error: Failed to open file: %s\n", argv[currentearly + 1]);
    return 2;
  }
  current = currentearly + 2;
  offset = table[index].offset;
  mode = 0;
  found = 0;
  if (table[index].mode) {
   if (! (list || listonlypro)) {
    if (argc <= current) {
      fprintf(stderr,"Error: No progress mode\n");
      return 3;
    }
   if (! comparecaps(argv[current], "LIST")) {
      listonlypro = 1;
      if (argc > (current + 1)) {
        fprintf(stderr, "Error: Listing mustn't modify configuration!\n");
        return 11;
      }
   } else {
    for (ii = 0; table[index].mode[ii]; ++ ii) {
      for (jj = 1; table[index].mode[ii][jj]; ++ jj) {
        if (! comparecaps(argv[current], table[index].mode[ii][jj])) {
          found = 1;
          break;
        }
      }
      if (found) { break; }
      ++ offset;
      ++ mode;
    }
    if (! found) {
      fprintf(stderr,"Error: Invalid progress mode: %s\n", argv[current]);
      return 3;
    }
    ++ current;
   }
   }
   if (list || listonlypro) {
    if (listonlypro) {
      ++ current;
    }
    if (argc > current) {
      fprintf(stderr, "Error: Listing mustn't modify configuration!\n");
      return 10;
    }
    offset += listpro;
    mode += listpro;
   }
  }
  offsetmod8 = offset & 7;
  offsetdiv8 = offset >> 3;
  bytes = table[index].bytes;
  bytesmask = 0xFFFFU >> (16 - bytes); /* 1 to 1, 2 to 3 */
  firstcurrent = current;
  for (; current < argc; ++ current) {
    unsigned rc;
    query = 0;
    rc = parsenumber(argv[current],
	current != firstcurrent,
	&want,
	&wantreplace, &wantplus, &wantminus);
    if (rc) return rc;
  }
  bufferbase = 0;
  fseek(ff, bufferbase, SEEK_SET);
  size = fread(buffer, 1, buffersize, ff);
  if (size < 32) {
    fprintf(stderr,"Error: lCFG block not found\n");
    return 5;
  }
  choices = (size / 16);
  for (found = 0, ii = 0; ii <= choices; ++ ii) {
    at = ii * 16;
    if (! memcmp(&buffer[at], "lCFGMULT""00", 10)) {
      amountblock = getu16(buffer + at + 10);
      if ((at + 16 + 16 * amountblock) > size) {
        fprintf(stderr,"Error: lCFG MULT block truncated"
          ", at=%u size=%u amountblock=%u\n", at, size, amountblock);
        return 12;
      }
      found = 1;
      break;
    }
  }
  if (found) {
    if (! allblocks && whichblock == 0) {
      if (! hintshown && amountblock > 1) {
        printf("Hint: Found %"PRIu16" lCFG blocks"
          ", using first named \"%.8s\"\n\n",
          amountblock, buffer + at + 16 + 16 * whichblock);
        hintshown = 1;
      }
      if (amountblock == 0) {
        fprintf(stderr,"Error: lCFG MULT block contains no pointers\n");
        return 13;
      }
      bufferbase = bufferbase + at + getu32(buffer + at + 16 + 8);
      bufferlength = getu16(buffer + at + 16 + 8 + 4);
    } else {
      if (allblocks) {
        if (whichblock == 0) {
          ++ whichblock;
        }
      }
      if (amountblock < whichblock) {
        fprintf(stderr,"Error: lCFG MULT block doesn't contain a pointer %u\n",
          whichblock);
        return 14;
      }
      bufferbase = bufferbase + at + getu32(buffer + at + 16 * whichblock + 8);
      bufferlength = getu16(buffer + at + 16 * whichblock + 8 + 4);
      if (! hintshown) {
        const char * sep = "";
        if (dosepbeforehint) {
          sep = "\n";
        }
        printf("%sOperating on lCFG block %u named \"%.8s\" range %"PRIu32"+%"PRIu16"\n\n",
          sep,
          whichblock, buffer + at + 16 * whichblock,
          bufferbase, bufferlength);
        hintshown = 1;
      }
    }
    if (bufferlength > buffersize) {
      fprintf(stderr,"Error: lCFG MULT block too large search space for %u\n",
        whichblock);
      return 15;
    }
    fseek(ff, bufferbase, SEEK_SET);
    size = fread(buffer, 1, bufferlength, ff);
    if (size < 32) {
      fprintf(stderr,"Error: lCFG block not found\n");
      return 16;
    }
  }
  choices = (size / 4) - 3;
  for (found = 0, ii = 0; ii <= choices; ++ ii) {
    at = ii * 4;
    if (! comparewild(buffer + at, table[index].sign, 8)) {
      found = 1;
      break;
    }
  }
  if (! found) {
    if (list || listonlypro || allblocks) {
      continue;
    }
    fprintf(stderr,"Error: lCFG block not found\n");
    return 6;
  }
  listfoundanycfg = 1;
  if ((bytesmask << (offsetmod8)) !=
  	(buffer[at + 8 + offsetdiv8] &  (bytesmask << (offsetmod8)) )) {
    if (list || listonlypro || allblocks) {
      continue;
    }
    fprintf(stderr,"Error: lCFG block found doesn't contain this configuration\n");
    return 8;
  }
  listfoundany = 1;
  listfoundanyone = 1;
  if (! query) {
    if (wantreplace) {
      buffer[at + offset] = want & 255;
      if (bytes == 2) {
        buffer[at + offset + 1] = want >> 8;
      }
    }
    buffer[at + offset] |= wantplus & 255;
    if (bytes == 2) {
      buffer[at + offset + 1] |= wantplus >> 8;
    }
    buffer[at + offset] &= (~wantminus) & 255;
    if (bytes == 2) {
      buffer[at + offset + 1] &= (~wantminus) >> 8;
    }
    fseek(ff, bufferbase, SEEK_SET);
    if (size != fwrite(buffer, 1, size, ff)) {
      fprintf(stderr,"Error: Failed to write file\n");
      return 9;
    }
    printf("Configuration updated\n");
  }
  {
    unsigned value;
    value = buffer[at + offset];
    if (bytes == 2) {
      value += buffer[at + offset + 1] * 256;
    }
    if (list) {
      const char * sep = "";
      if (dosepbeforelist) {
        sep = "\n";
      }
      dosepbeforelist = 1;
      printf("%s%s: ", sep, table[index].keywords[0]);
    }
    printf("Current mode = %u = %0*Xh\n", value, bytes * 2, value);
  if (table[index].types == 1) {
    const char *pp;
    const char *flag16 = "";
    unsigned progress = value;
    if (mode != 2 && (progress & 16) ) {
      progress &= ~16;
      flag16 = ", force to file too";
    }
    if (progress < flagsamount) {
      pp = names[progress];
    } else {
      pp = "Unknown choice";
    }
    printf("%s progress choice is %u (%s%s)\n",
	modekeywords_progress[mode][0], value, pp, flag16);
  } else if (table[index].types == 2) {
    const char *pp = "";
    const char *qq = "";
    unsigned choice = value;
    if (choice == 0) {
      pp = " (Immediate timeout)";
    }
    if (choice == 255) {
      pp = " (Infinite timeout)";
    }
    if (choice != 1) {
      qq = "s";
    }
    printf("Menu timeout is %u second%s%s\n",
	value, qq, pp);
  } else if (table[index].types == 3) {
    const char *pp;
    unsigned choice = value;
    if (choice < flagsamount) {
      pp = names[choice];
    } else {
      pp = "Unknown choice";
    }
    printf("Menu default choice is %u (%s)\n",
	value, pp);
  } else {
    for (ii = 0; ii < flagsamount; ++ ii) {
      if (ii + 1 < flagsamount && !names[ii + 1]) {
        printf("%s is %s\n", names[ii], extnames[(value >> ii) & 3]);
      } else if (!names[ii]) {
      } else if (!*names[ii]) {
        if (value & (1 << ii)) {
          printf("Unknown flag %u is ON\n", 1 << ii);
        }
      } else {
        printf("%s is %s\n", names[ii], value & (1 << ii) ? "ON" : "OFF");
      }
    }
    for (; ii < (bytes * 8); ++ ii) {
      if (value & (1 << ii)) {
        printf("Unknown flag %u is ON\n", 1 << ii);
      }
    }
  }
  }
 }
  if (allblocks) {
    if (whichblock < amountblock) {
      ++ whichblock;
      moreblocks = 1;
      dosepbeforehint = !!listfoundanyone;
      listfoundanyone = 0;
      dosepbeforelist = 0;
    } else {
      moreblocks = 0;
    }
  }
 }
  if (argc < (2 + currentearly) && list) {
    printf("\n");
    return patchqrymain(1, argv + 1);
  }
  if (argc == (2 + currentearly) && list) {
    printf("\n");
    return patchqrymain(argc - 1, argv + 1);
  }
  if (argc >= (2 + currentearly) && (list || listonlypro || allblocks) && 0 == listfoundanycfg) {
    fprintf(stderr, "Error: lCFG block not found\n");
    return 17;
  } else if (argc >= (2 + currentearly) && (list || listonlypro || allblocks) && 0 == listfoundany) {
    printf("lCFG block does not contain any known configuration\n");
  }
  return 0;
}

